package Future::Batch::XS;

use 5.014;
use strict;
use warnings;

use Future;
use Exporter 'import';

our $VERSION = '0.01';
our @EXPORT_OK = qw(batch);

require XSLoader;
XSLoader::load('Future::Batch::XS', $VERSION);

1;

__END__

=head1 NAME

Future::Batch::XS - Full XS batch processing for Future operations

=head1 VERSION

Version 0.01

=head1 SYNOPSIS

    use Future::Batch::XS qw(batch);
    
    my $results = batch(
        items      => \@urls,
        concurrent => 5,
        worker     => sub {
            my ($url, $index) = @_;
            return $http->GET($url);
        },
    )->get;

=head1 DESCRIPTION

Future::Batch::XS provides a complete XS implementation of batch processing
for Future operations. Unlike typical XS modules that wrap Perl code, this
module implements the entire batch processing loop in C/XS.

Key features:

=over 4

=item * 100% XS implementation - no Perl code in the hot path

=item * Batch state stored in C struct with SV magic

=item * XS closures for Future callbacks using CvXSUBANY

=item * Memory-efficient callback data management

=item * Full async support via IO::Async::Loop

=back

=head1 METHODS

=head2 new

    my $batch = Future::Batch::XS->new(%options);

Constructor. Options:

=over 4

=item * C<concurrent> - Maximum concurrent operations (default: 10)

=item * C<fail_fast> - Stop on first failure (default: false)

=item * C<on_progress> - Progress callback coderef

=item * C<loop> - IO::Async::Loop for async scheduling

=back

=head2 run

    my $future = $batch->run(items => \@items, worker => \&worker);

Execute batch processing. Returns a Future.

=head2 concurrent

    my $n = $batch->concurrent;
    $batch->concurrent(20);

Get/set maximum concurrent operations.

=head2 fail_fast

    my $ff = $batch->fail_fast;
    $batch->fail_fast(1);

Get/set fail-fast mode.

=head2 on_progress

    $batch->on_progress(sub { my ($done, $total) = @_; ... });

Get/set progress callback.

=head2 loop

    $batch->loop($loop);

Get/set IO::Async::Loop.

=head1 FUNCTIONS

=head2 batch

    my $future = batch(
        items      => \@items,
        worker     => \&worker,
        concurrent => 5,
    );

Functional interface combining new() and run().

=head1 IMPLEMENTATION NOTES

This module creates XS closures for Future's on_done/on_fail callbacks.
Each callback stores its closure data (state reference, item, index) in
a C struct attached to the CV via SV magic (PERL_MAGIC_ext).

The callback data is automatically freed when the CV is garbage collected.

=head1 SEE ALSO

L<Future::Batch>, L<Future>, L<Future::XS>

=head1 AUTHOR

Your Name E<lt>your@email.comE<gt>

=head1 LICENSE

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
