#!/usr/bin/env perl
use strict;
use warnings;

use Test::More;
use Math::Prime::Util::GMP qw/is_powerfree is_square_free
                              next_powerfree prev_powerfree
                              powerfree_count nth_powerfree/;

my @simple = (0 .. 16,
              758096738,434420340,870589313,695486396,602721315,418431087,
              752518565,723570005,506916483,617459403);

plan tests => 3       # is_powerfree, is square free
            + 5+2+3   # next_powerfree, prev_powerfree
            + 11+4+1  # powerfree_count
            + 7+1     # nth_powerfree
            + 0;

##### is_powerfree

is_deeply( [map { is_powerfree($_)   } @simple],
           [map { is_square_free($_) } @simple],
           "is_powerfree(n) matches is_square_free(n)" );
is_deeply( [map { is_powerfree($_)   } @simple],
           [0,1,1,1,0,1,1,1,0,0,1,1,0,1,1,1,0,0,0,0,0,0,0,1,1,1,1],
           "is_powerfree(n) works for simple inputs" );
is_deeply( [map { is_powerfree($_,3) } @simple],
           [0,1,1,1,1,1,1,1,0,1,1,1,1,1,1,1,0,1,1,1,1,0,1,1,1,1,1],
           "is_powerfree(n,3) works for simple inputs" );

##### next_powerfree / prev_powerfree

is_deeply( [map { next_powerfree($_) } 22018,22019,22026],
           [22019, 22019+7, 22019+7+1],
           "next_powerfree(22018 ...,2)" );
is_deeply( [map { next_powerfree($_) } 8870022,8870023,8870033],
           [8870023, 8870023+10, 8870023+10+1],
           "next_powerfree(8870023 ...,2)" );
is_deeply( [map { next_powerfree($_) } qw/125781000834058566 125781000834058567 125781000834058586/],
           [qw/125781000834058567 125781000834058586 125781000834058589/],
           "next_powerfree(125781000834058586 ...,2)" );
is_deeply( [map { next_powerfree($_) } qw/12345678901234567887 12345678901234567898 12345678901234567923 12345678901234568327 12345678901234582095 12345678901235019469 12345678901237442923/],
           [qw/12345678901234567891 12345678901234567901 12345678901234567927 12345678901234568333 12345678901234582102 12345678901235019477 12345678901237442933/],
           "next_powerfree(12345678901234567889 .., 2)" );
is_deeply( [map { prev_powerfree($_) } qw/12345678901234567891 12345678901234567901 12345678901234567927 12345678901234568333 12345678901234582102 12345678901235019477 12345678901237442933/],
           [qw/12345678901234567887 12345678901234567898 12345678901234567923 12345678901234568327 12345678901234582095 12345678901235019469 12345678901237442923/],
           "prev_powerfree(12345678901234567889 .., 2)" );

is_deeply( [map { next_powerfree($_,3) } qw/1 7 79 1374 22623 18035621 4379776619 1204244328623 6163278473258246 2604639091138247/],
           [qw/2 9 82 1378 22628 18035627 4379776626 1204244328631 6163278473258255 2604639091138257/],
           "next_powerfree(1,7,79,1374,... ,3)" );
is_deeply( [map { prev_powerfree($_,3) } qw/2 9 82 1378 22628 18035627 4379776626 1204244328631 6163278473258255 2604639091138257/],
           [qw/1 7 79 1374 22623 18035621 4379776619 1204244328623 6163278473258246 2604639091138247/],
           "prev_powerfree(1,7,79,1374,... ,3)" );

is(next_powerfree("12345678901234629871",3),"12345678901234629876","next_powerfree(12345678901234629871,3)");
is(next_powerfree("12345678901237158124",4),"12345678901237158129","next_powerfree(12345678901237158124,4)");
is(next_powerfree("12345678901239790623",5),"12345678901239790627","next_powerfree(12345678901239790623,5)");


##### powerfree_count and powerfree_sum

foreach my $k (0..10) {
    my $n = 100;
    is_deeply(
        [map { powerfree_count($_, $k) } 0..$n],
        [map { scalar grep { is_powerfree($_, $k) } 1..$_ } 0..$n],
        "powerfree_count(0..$n, $k)"
    );
}

is( powerfree_count(12345,2),  7503, "powerfree_count(12345,2) = 7503");
is( powerfree_count(12345,3), 10272, "powerfree_count(12345,3) = 10272");
is( powerfree_count(12345,4), 11408, "powerfree_count(12345,4) = 11408");
is( powerfree_count(123456,32), 123456, "powerfree_count(123456,32) = 123456");

is_deeply( [map {powerfree_count("1234567890123456789012345678901",$_)} 6..14],
           [qw/1213521707923465541406191918053 1224345490016006922209431194227 1229554558224768315813401074308 1232093362652841310357556358984 1233341239602435880533138901815 1233958082101321205441537578995 1234264154311850512659828305857 1234416410753387417554556681046 1234492279773576288583884594288/],
           "powerfree_count(1234567890123456789012345678901, 6..14)" );

##### nth_powerfree
is(nth_powerfree(7503), 12345, "nth_powerfree(7503) = 12345");
is(nth_powerfree(10272,3), 12345, "nth_powerfree(10272,3) = 12345");
is(nth_powerfree(11408,4), 12345, "nth_powerfree(11408,4) = 12345");
is(nth_powerfree(915099,3), 1099999, "nth_powerfree(915099,3) = 1099999");
is(nth_powerfree("1000000",2), 1644918, "nth_powerfree(10^6,2) = 1644918");
is(nth_powerfree("1000000",3), 1202057, "nth_powerfree(10^6,3) = 1202057");
is(nth_powerfree("100000000",5), 103692775, "nth_powerfree(10^8,5) = 103692775");

is_deeply( [map {nth_powerfree("100000000000000000000",$_)} 4..20],
           [qw/108232323371113819066 103692775514336992646 101734306198444913972 100834927738192282689 100407735619794433943 100200839282608221439 100099457512781808527 100049418860411946451 100024608655330804829 100012271334757848914 100006124813505870482 100003058823630702047 100001528225940865186 100000763719763789974 100000381729326499982 100000190821271655392 100000095396203387279/],
           "nth_powerfree(10^20, 4..20)" );
